import asyncio
import json
import time

from pylog.pylogger import PyLogger

from py_pli.pylib import GlobalVar
from py_pli.pylib import send_msg
from py_pli.pyexception import UrpcFirmwareException

from urpc_enum.error import MoverErrorCode
from urpc_enum.stackerliftmoverparameter import StackerLiftMoverParameter

from fleming.common.firmware_util import *

from fleming.rbartz.stacker_test_config import frame_sensor_status
from fleming.rbartz.stacker_test_config import stack_sensor_status

LIFT_USTEPS = 256
RELEASE_USTEPS = 64

LIFT_MOVER_USTEPS_PER_MM = 1.0 / 0.03175 * LIFT_USTEPS
RELEASE_MOVER_USTEPS_PER_MM = 1.0 / 0.042 * RELEASE_USTEPS

def get_lift_usteps(millimeter) -> int:
    return round(millimeter * LIFT_MOVER_USTEPS_PER_MM)

def get_release_usteps(millimeter) -> int:
    return round(millimeter * RELEASE_MOVER_USTEPS_PER_MM)

# Reference Positions
LIFT_MOVER_HOME         = get_lift_usteps(-9.5)
STACKER_TABLE_SURFACE   = get_lift_usteps(-4.3)
STACKER_FRAME_SURFACE   = get_lift_usteps(34.6)
RELEASE_MOVER_HOME      = get_release_usteps(-5.3)
MAGAZINE_LOCK_SURFACE   = get_release_usteps(38.2)
RELEASE_MOVER_HALF_OPEN = get_release_usteps(45.5)
RELEASE_MOVER_OPEN      = get_release_usteps(50.5)

# Constant Offsets
PHASE_OFFSET        = get_lift_usteps(13.0)
GRIP_OFFSET         = get_lift_usteps( 8.0)
GRIP_SAFETY_MARGIN  = get_lift_usteps( 2.0)     # EnSight SAFETY_MARGIN_FOR_GRIP_POS: 200 [1/100*mm]
LOCK_OFFSET         = get_release_usteps(3.0)

# Lift Mover Positions
LIFT_POS = {
    'home'      : LIFT_MOVER_HOME,                          # EnSight LiftCpDist: -950 [1/100*mm]
    'phase'     : STACKER_TABLE_SURFACE + PHASE_OFFSET,     # EnSight LiftPhasePos: 950 [1/100*mm]
    'grip'      : STACKER_FRAME_SURFACE + GRIP_OFFSET,      # EnSight LiftGripPos: 4300 [1/100*mm]
    'lift_stack': STACKER_FRAME_SURFACE + GRIP_OFFSET + GRIP_SAFETY_MARGIN,
}

# Release Mover Positions
RELEASE_POS = {
    'home'      : RELEASE_MOVER_HOME,                   # EnSight ReleaseCpDist: -480 [1/100*mm]
    'lock'      : MAGAZINE_LOCK_SURFACE + LOCK_OFFSET,  # EnSight ReleaseStackLockPos: 4050 [1/100*mm]
    'half_open' : RELEASE_MOVER_HALF_OPEN,              # EnSight ReleaseHalfOpenPos: 4600 [1/100*mm]
    'open'      : RELEASE_MOVER_OPEN,                   # EnSight ReleaseOpenPos: 5000 [1/100*mm]
}

# EnSight liftSlowProfile: 3000, 100, 40, 40 (speed[1/100*mm/s], drivePower[%], accel[10*mm/s^2], decel[10*mm/s^2])
LIFT_SLOW_PROFILE = 2
LIFT_SLOW_SPEED = get_lift_usteps( 15.0)
LIFT_SLOW_ACCEL = get_lift_usteps(400.0)

LIFT_PROFILE = 1
LIFT_SPEED = get_lift_usteps( 30.0)
LIFT_ACCEL = get_lift_usteps(400.0)

LIFT_HOME_PROFILE = 0

# EnSight releaseProfile: 1500, 40, 14, 14 (speed[1/100*mm/s], drivePower[%], accel[10*mm/s^2], decel[10*mm/s^2])
RELEASE_PROFILE = 1
RELEASE_SPEED = get_release_usteps( 15.0)
RELEASE_ACCEL = get_release_usteps(140.0)

RELEASE_HOME_PROFILE = 0


PLATE_EXPECTED_NO  = 0x00
PLATE_EXPECTED_YES = 0x01
STOP_ON_FAIL       = 0x02

PLATE_EDGE_HEIGHT = get_lift_usteps( 5.0)   # OptiPlate - 384:  3.0 mm + 2 mm safety
PLATE_HEIGHT      = get_lift_usteps(14.4)   # OptiPlate - 384: 14.4 mm

PLATE_SAMPLE_TIME_MS = 20
PLATE_DELTA_LIMIT    = 0.05
PLATE_MEAN_LIMIT     = 0.05
PLATE_SENSOR_CURRENT = 0.1


# Stacker Utility Functions ####################################################

def get_lift_mover(side):
    if side == 'l' or side == 'left':
        return get_stacker_lift_mover_endpoint('llm')
    if side == 'r' or side == 'right':
        return get_stacker_lift_mover_endpoint('rlm')

    raise ValueError(f"Invalid side.")


def get_release_mover(side):
    if side == 'l' or side == 'left':
        return get_mover_endpoint('lrm')
    if side == 'r' or side == 'right':
        return get_mover_endpoint('rrm')

    raise ValueError(f"Invalid side.")


def get_gripper_pwm(side):
    if side == 'l' or side == 'left':
        return 0
    if side == 'r' or side == 'right':
        return 1

    raise ValueError(f"Invalid side.")


def frame_sensor_callback_left(type):
    frame_sensor_status['left'] = int(type == 1)


def frame_sensor_callback_right(type):
    frame_sensor_status['right'] = int(type == 1)


def stack_sensor_callback_left(type):
    stack_sensor_status['left'] = int(type == 1)


def stack_sensor_callback_right(type):
    stack_sensor_status['right'] = int(type == 1)


def calculate_lift_move_time(distance, speed=LIFT_SPEED, accel=LIFT_ACCEL):
    accel_time = speed / accel
    accel_dist = speed * accel_time / 2
    if (accel_dist > distance):
        raise Exception(f"Distance must be greater than accel distance.")
    
    drive_time = (distance - accel_dist) / speed
    return round((accel_time + drive_time) * 1000)


def calculate_lift_move_time_from_mm(distance):
    distance = get_lift_usteps(distance)
    return f"{calculate_lift_move_time(distance):.3f} ms"


# Stacker Test Functions #######################################################

async def stk_init():

    await send_msg(json.dumps({'result': f"Initializing stacker..."}))
    
    PyLogger.logger.info(f"Start MC6 firmware")
    await start_firmware('mc6_stk')

    PyLogger.logger.info(f"Initialize Left Lift Mover")
    llm = get_stacker_lift_mover_endpoint('llm')
    await llm.SetProfile(
        handle=LIFT_SLOW_PROFILE, speed=LIFT_SLOW_SPEED, accel=LIFT_SLOW_ACCEL, decel=LIFT_SLOW_ACCEL, uSteps=LIFT_USTEPS, drivePower=154, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await llm.SetProfile(
        handle=LIFT_PROFILE, speed=LIFT_SPEED, accel=LIFT_ACCEL, decel=LIFT_ACCEL, uSteps=LIFT_USTEPS, drivePower=60, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await llm.SetProfile(
        handle=LIFT_HOME_PROFILE, speed=LIFT_SPEED, accel=LIFT_ACCEL, decel=LIFT_ACCEL, uSteps=LIFT_USTEPS, drivePower=60, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await llm.UseProfile(LIFT_PROFILE)
    await llm.SetParameter(StackerLiftMoverParameter.HomeSearchDirection,            0)
    await llm.SetParameter(StackerLiftMoverParameter.HomeMaxDistance,          1000000)
    await llm.SetParameter(StackerLiftMoverParameter.HomeMaxReverseDistance,   1000000)
    await llm.SetParameter(StackerLiftMoverParameter.HomeExtraReverseDistance,       0)
    await llm.SetParameter(StackerLiftMoverParameter.HomeCalibrationSpeed,       10000)
    await llm.SetParameter(StackerLiftMoverParameter.HomePosition,    LIFT_POS['home'])
    await llm.SetParameter(StackerLiftMoverParameter.HomeSensorEnable,            0x01)
    await llm.SetParameter(StackerLiftMoverParameter.MovementDirection,              0)
    await llm.SetConfigurationStatus(1)

    await llm.SetParameter(StackerLiftMoverParameter.PlateScanSampleTime, PLATE_SAMPLE_TIME_MS)
    await llm.SetParameter(StackerLiftMoverParameter.PlateScanDeltaLimit, PLATE_DELTA_LIMIT)
    await llm.SetParameter(StackerLiftMoverParameter.PlateScanMeanLimit, PLATE_MEAN_LIMIT)
    await llm.SetParameter(StackerLiftMoverParameter.PlateSensorCurrent, PLATE_SENSOR_CURRENT)
    await llm.SetParameter(StackerLiftMoverParameter.FrameSensorEventEnable, 3)
    await llm.SetParameter(StackerLiftMoverParameter.StackSensorEventEnable, 3)

    frame_sensor_status['left'] = (await llm.GetParameter(StackerLiftMoverParameter.FrameSensorStatus))[0]
    llm.subscribeSendFrameSensorEvent(frame_sensor_callback_left)

    stack_sensor_status['left'] = (await llm.GetParameter(StackerLiftMoverParameter.StackSensorStatus))[0]
    llm.subscribeSendStackSensorEvent(stack_sensor_callback_left)

    PyLogger.logger.info(f"Initialize Right Lift Mover")
    rlm = get_stacker_lift_mover_endpoint('rlm')
    await rlm.SetProfile(
        handle=LIFT_SLOW_PROFILE, speed=LIFT_SLOW_SPEED, accel=LIFT_SLOW_ACCEL, decel=LIFT_SLOW_ACCEL, uSteps=LIFT_USTEPS, drivePower=154, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await rlm.SetProfile(
        handle=LIFT_PROFILE, speed=LIFT_SPEED, accel=LIFT_ACCEL, decel=LIFT_ACCEL, uSteps=LIFT_USTEPS, drivePower=60, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await rlm.SetProfile(
        handle=LIFT_HOME_PROFILE, speed=LIFT_SPEED, accel=LIFT_ACCEL, decel=LIFT_ACCEL, uSteps=LIFT_USTEPS, drivePower=60, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
    )
    await rlm.UseProfile(LIFT_PROFILE)
    await rlm.SetParameter(StackerLiftMoverParameter.HomeSearchDirection,            0)
    await rlm.SetParameter(StackerLiftMoverParameter.HomeMaxDistance,          1000000)
    await rlm.SetParameter(StackerLiftMoverParameter.HomeMaxReverseDistance,   1000000)
    await rlm.SetParameter(StackerLiftMoverParameter.HomeExtraReverseDistance,       0)
    await rlm.SetParameter(StackerLiftMoverParameter.HomeCalibrationSpeed,       10000)
    await rlm.SetParameter(StackerLiftMoverParameter.HomePosition,    LIFT_POS['home'])
    await rlm.SetParameter(StackerLiftMoverParameter.HomeSensorEnable,            0x01)
    await rlm.SetParameter(StackerLiftMoverParameter.MovementDirection,              0)
    await rlm.SetConfigurationStatus(1)

    await rlm.SetParameter(StackerLiftMoverParameter.PlateScanSampleTime, PLATE_SAMPLE_TIME_MS)
    await rlm.SetParameter(StackerLiftMoverParameter.PlateScanDeltaLimit, PLATE_DELTA_LIMIT)
    await rlm.SetParameter(StackerLiftMoverParameter.PlateScanMeanLimit, PLATE_MEAN_LIMIT)
    await rlm.SetParameter(StackerLiftMoverParameter.PlateSensorCurrent, PLATE_SENSOR_CURRENT)
    await rlm.SetParameter(StackerLiftMoverParameter.FrameSensorEventEnable, 3)
    await rlm.SetParameter(StackerLiftMoverParameter.StackSensorEventEnable, 3)

    frame_sensor_status['right'] = (await rlm.GetParameter(StackerLiftMoverParameter.FrameSensorStatus))[0]
    rlm.subscribeSendFrameSensorEvent(frame_sensor_callback_right)

    stack_sensor_status['right'] = (await rlm.GetParameter(StackerLiftMoverParameter.StackSensorStatus))[0]
    rlm.subscribeSendStackSensorEvent(stack_sensor_callback_right)

    PyLogger.logger.info(f"Initialize Left Release Mover")
    lrm = get_mover_endpoint('lrm')
    await lrm.SetProfile(
        handle=RELEASE_PROFILE, speed=RELEASE_SPEED, accel=RELEASE_ACCEL, decel=RELEASE_ACCEL, uSteps=RELEASE_USTEPS, drivePower=25, holdPower=1, drivePowerHoldTime=200, drivePowerFallTime=1000
    )
    await lrm.SetProfile(
        handle=RELEASE_HOME_PROFILE, speed=RELEASE_SPEED, accel=RELEASE_ACCEL, decel=RELEASE_ACCEL, uSteps=RELEASE_USTEPS, drivePower=25, holdPower=1, drivePowerHoldTime=200, drivePowerFallTime=1000
    )
    await lrm.UseProfile(RELEASE_PROFILE)
    await lrm.SetParameter(StackerLiftMoverParameter.HomeSearchDirection,            0)
    await lrm.SetParameter(StackerLiftMoverParameter.HomeMaxDistance,          1000000)
    await lrm.SetParameter(StackerLiftMoverParameter.HomeMaxReverseDistance,   1000000)
    await lrm.SetParameter(StackerLiftMoverParameter.HomeExtraReverseDistance,       0)
    await lrm.SetParameter(StackerLiftMoverParameter.HomeCalibrationSpeed,       10000)
    await lrm.SetParameter(StackerLiftMoverParameter.HomePosition, RELEASE_POS['home'])
    await lrm.SetParameter(StackerLiftMoverParameter.HomeSensorEnable,            0x01)
    await lrm.SetParameter(StackerLiftMoverParameter.MovementDirection,              1)
    await lrm.SetConfigurationStatus(1)

    PyLogger.logger.info(f"Initialize Right Release Mover")
    rrm = get_mover_endpoint('rrm')
    await rrm.SetProfile(
        handle=RELEASE_PROFILE, speed=RELEASE_SPEED, accel=RELEASE_ACCEL, decel=RELEASE_ACCEL, uSteps=RELEASE_USTEPS, drivePower=25, holdPower=1, drivePowerHoldTime=200, drivePowerFallTime=1000
    )
    await rrm.SetProfile(
        handle=RELEASE_HOME_PROFILE, speed=RELEASE_SPEED, accel=RELEASE_ACCEL, decel=RELEASE_ACCEL, uSteps=RELEASE_USTEPS, drivePower=25, holdPower=1, drivePowerHoldTime=200, drivePowerFallTime=1000
    )
    await rrm.UseProfile(RELEASE_PROFILE)
    await rrm.SetParameter(StackerLiftMoverParameter.HomeSearchDirection,            0)
    await rrm.SetParameter(StackerLiftMoverParameter.HomeMaxDistance,          1000000)
    await rrm.SetParameter(StackerLiftMoverParameter.HomeMaxReverseDistance,   1000000)
    await rrm.SetParameter(StackerLiftMoverParameter.HomeExtraReverseDistance,       0)
    await rrm.SetParameter(StackerLiftMoverParameter.HomeCalibrationSpeed,       10000)
    await rrm.SetParameter(StackerLiftMoverParameter.HomePosition, RELEASE_POS['home'])
    await rrm.SetParameter(StackerLiftMoverParameter.HomeSensorEnable,            0x01)
    await rrm.SetParameter(StackerLiftMoverParameter.MovementDirection,              1)
    await rrm.SetConfigurationStatus(1)

    PyLogger.logger.info(f"Initialize Left Gripper")
    node = get_node_endpoint('mc6_stk')
    # DC-12V ED:100% -> 50% for 24V
    await node.EnablePWMOutput(number=0, enable=0)
    await node.ConfigurePWMOutput(number=0, switchCurrent=500, holdCurrent=0, switchTime=0)

    PyLogger.logger.info(f"Initialize Right Gripper")
    node = get_node_endpoint('mc6_stk')
    # DC-12V ED:100% -> 50% for 24V
    await node.EnablePWMOutput(number=1, enable=0)
    await node.ConfigurePWMOutput(number=1, switchCurrent=500, holdCurrent=0, switchTime=0)

    return f"stk_init() done"


async def stk_home(side):
    lift_mover = get_lift_mover(side)
    release_mover = get_release_mover(side)

    await send_msg(json.dumps({'result': f"Homing {side} stacker..."}))
    
    (rm_error,) = await release_mover.Home()
    (lm_error,) = await lift_mover.Home()
    return (lm_error, rm_error)


async def stk_home_all():
    pos_error = await asyncio.gather(
        stk_home('left'),
        stk_home('right'),
    )
    return f"left: {pos_error[0]}, right: {pos_error[1]}"


async def stk_lm_move(side, position):
    mover = get_lift_mover(side)
    if position in LIFT_POS:
        position = LIFT_POS[position]
    else:
        position = get_lift_usteps(position)

    await mover.Move(position)
    return f"stk_lm_move() done"


async def stk_lm_home(side):
    mover = get_lift_mover(side)
    return await mover.Home()


async def stk_rm_move(side, position):
    mover = get_release_mover(side)
    if position in RELEASE_POS:
        position = RELEASE_POS[position]
    else:
        position = get_release_usteps(position)

    await mover.Move(position)
    return f"stk_rm_move() done"


async def stk_rm_home(side):
    mover = get_release_mover(side)
    return await mover.Home()


def stk_get_frame_sensor_status(side):
    if side == 'l' or side == 'left':
        return frame_sensor_status['left']
    if side == 'r' or side == 'right':
        return frame_sensor_status['right']

    raise ValueError(f"Invalid side.")


def stk_get_stack_sensor_status(side):
    if side == 'l' or side == 'left':
        return stack_sensor_status['left']
    if side == 'r' or side == 'right':
        return stack_sensor_status['right']

    raise ValueError(f"Invalid side.")


async def stk_test_plate_sensor(side):
    lift_mover = get_lift_mover(side)
    (signal, base) = await lift_mover.TestPlateSensor()
    return f"{signal} - {base} = {signal - base}"


async def stk_has_plate(side):
    lift_mover = get_lift_mover(side)
    (signal, base) = await lift_mover.TestPlateSensor()
    return (signal - base) < PLATE_MEAN_LIMIT


async def stk_run_plate_scan(side, start_pos='home', end_pos='phase'):
    lift_mover = get_lift_mover(side)

    await send_msg(json.dumps({'result': f"Running plate scan..."}))

    if start_pos in LIFT_POS:
        start_pos = LIFT_POS[start_pos]
    else:
        start_pos = get_lift_usteps(start_pos)

    if end_pos in LIFT_POS:
        end_pos = LIFT_POS[end_pos]
    else:
        end_pos = get_lift_usteps(end_pos)

    # Move to start position.
    await lift_mover.Move(start_pos)
    # Move to end position and run plate scan.
    start_time = 0
    end_time   = calculate_lift_move_time(end_pos - start_pos)
    await lift_mover.SetupPlateScan(startTime=start_time, endTime=end_time, options=PLATE_EXPECTED_YES)
    await lift_mover.Move(end_pos)

    return f"stk_run_plate_scan() done"


async def stk_plate_sensor_test():
    llm = get_lift_mover('left')
    rlm = get_lift_mover('right')

    await send_msg(json.dumps({'result': f"Running plate sensor test..."}))

    await stk_init()
    await stk_home_all()

    result_left = 'passed'
    try:
        await stk_run_plate_scan('left', end_pos=20.0)
    except UrpcFirmwareException as ex:
        if ex.errorCode != MoverErrorCode.PlateScanFailed:
            raise

    (delta_left, mean_left) = await llm.GetPlateScanResult()
    if (delta_left <= PLATE_DELTA_LIMIT) or (mean_left >= PLATE_MEAN_LIMIT):
        result_left = 'failed'

    result_right = 'passed'
    try:
        await stk_run_plate_scan('right', end_pos=20.0)
    except UrpcFirmwareException as ex:
        if ex.errorCode != MoverErrorCode.PlateScanFailed:
            raise

    (delta_right, mean_right) = await rlm.GetPlateScanResult()
    if (delta_right <= PLATE_DELTA_LIMIT) or (mean_right >= PLATE_MEAN_LIMIT):
        result_right = 'failed'
    
    await send_msg(json.dumps({'result': f"side ; delta ; mean ; result"}))
    await send_msg(json.dumps({'result': f"left ; {delta_left:.3f} ; {mean_left:.3f} ; {result_left}"}))
    await send_msg(json.dumps({'result': f"right ; {delta_right:.3f} ; {mean_right:.3f} ; {result_right}"}))

    return f"stk_plate_sensor_test() done"
    

async def stk_get_plate_scan_result(side):
    lift_mover = get_lift_mover(side)
    (delta, mean) = await lift_mover.GetPlateScanResult()
    return f"delta: {delta} mean: {mean}"


async def stk_read_plate_scan_buffer(side):
    lift_mover = get_lift_mover(side)
    plate_scan_data = []
    for index in range(0, 512, 29):
        size = min(29, (512 - index))
        plate_scan_data.extend(await lift_mover.ReadPlateScanBuffer(index, size))
    
    return plate_scan_data


async def stk_print_plate_scan(side):
    plate_scan_data = await stk_read_plate_scan_buffer(side)
    point = [0]*256
    for i in range(256):
        point[i] = plate_scan_data[i * 2 + 1] - plate_scan_data[i * 2]
        point[i] = round(point[i] / 2**31 * 100 / 2)
    
    chart = [['\u25FB']*100 for i in range(11)]
    for x in range(100):
        y = 10 - min(point[x], 10)
        chart[y][x] = '\u25FC'
    
    for y in range(11):
        await send_msg(json.dumps({'result': ''.join(chart[y])}))
        PyLogger.logger.info(''.join(chart[y]))


async def stk_run_parallel(iterations=1, delay=0):
    if not await stk_has_plate('left'):
        await stk_get_plate('left')

    if await stk_has_plate('right'):
        await stk_put_plate('right')

    for i in range(iterations):
        await asyncio.sleep(delay)
        results = await asyncio.gather(
            stk_put_plate('left'),
            stk_get_plate('right'),
        )
        PyLogger.logger.info(f"{i+1:3d} Left:  {results[0]}")
        PyLogger.logger.info(f"{i+1:3d} Right: {results[1]}")
        await asyncio.sleep(delay)
        results = await asyncio.gather(
            stk_get_plate('left'),
            stk_put_plate('right'),
        )
        PyLogger.logger.info(f"{i+1:3d} Left:  {results[0]}")
        PyLogger.logger.info(f"{i+1:3d} Right: {results[1]}")

    return f"stk_run_parallel done"


async def stk_get_plate_all():
    results = await asyncio.gather(
        stk_get_plate('left'),
        stk_get_plate('right'),
    )
    return results


async def stk_get_plate(side):
    duration = 0.0
    duration += await stk_get_p1(side)
    duration += await stk_get_p2(side)
    return f"stk_get_plate() done after {duration:.3f}s"


async def stk_get_p1(side):
    mc6_stk = get_node_endpoint('mc6_stk')
    lift_mover = get_lift_mover(side)
    release_mover = get_release_mover(side)
    gripper_pwm = get_gripper_pwm(side)

    tickstart = time.perf_counter()

    await send_msg(json.dumps({'result': f"Get {side} plate phase 1..."}))

    # Check that the lift mover is at the home position.    #TODO A more sophisticated method to ensure the correct stacker state is adviced.
    (position,) = await lift_mover.GetPosition()
    if position != LIFT_POS['home']:
        raise Exception(f"Lift mover not at home position")
    
    # Lock the stack and the hooks in there thus ensuring that the plates are firmly in the stack.
    if not stk_get_stack_sensor_status(side):
        raise Exception("No stack present.")
    
    await release_mover.Move(RELEASE_POS['lock'])

    # Move below the stack.
    await lift_mover.UseProfile(LIFT_PROFILE)
    start_time = 0
    end_time   = calculate_lift_move_time(LIFT_POS['phase'] - LIFT_POS['home'])
    await lift_mover.SetupPlateScan(startTime=start_time, endTime=end_time, options=(PLATE_EXPECTED_NO | STOP_ON_FAIL))
    await lift_mover.Move(LIFT_POS['grip'] - GRIP_SAFETY_MARGIN)

    # Lift the stack.
    await lift_mover.UseProfile(LIFT_SLOW_PROFILE)
    await lift_mover.Move(LIFT_POS['lift_stack'])

    # Gripper Close.
    await mc6_stk.EnablePWMOutput(gripper_pwm, 1)
    
    # Open the hooks in the stack.
    await release_mover.Move(RELEASE_POS['open'])
    # Move the plate below the hooks of the stack.
    await lift_mover.Move(LIFT_POS['grip'] - PLATE_EDGE_HEIGHT)
    # Put the hooks to semiopen state above the plate's edge to ensure that only one plate comes down.
    await release_mover.Move(RELEASE_POS['half_open'])
    # Move the plate below the stack.
    await lift_mover.Move(LIFT_POS['grip'] - PLATE_HEIGHT - GRIP_SAFETY_MARGIN)
    
    # Move the plate to the waiting phase.
    await lift_mover.UseProfile(LIFT_PROFILE)
    await lift_mover.Move(LIFT_POS['phase'])
    # Gripper Open.
    await mc6_stk.EnablePWMOutput(gripper_pwm, 0)

    # Fully close the hooks in the stack.
    await release_mover.Move(RELEASE_POS['lock'])

    return (time.perf_counter() - tickstart)
    
    
async def stk_get_p2(side):
    lift_mover = get_lift_mover(side)

    tickstart = time.perf_counter()

    await send_msg(json.dumps({'result': f"Get {side} plate phase 2..."}))
    
    # Check that the lift mover is at the phase position.   #TODO A more sophisticated method to ensure the correct stacker state is adviced.
    (position,) = await lift_mover.GetPosition()
    if position != LIFT_POS['phase']:
        raise Exception(f"Lift mover must be at phase position")

    # Move the plate to the plate carrier.
    await lift_mover.UseProfile(LIFT_PROFILE)
    start_time = 0
    end_time   = calculate_lift_move_time(LIFT_POS['phase'] - LIFT_POS['home'])
    await lift_mover.SetupPlateScan(startTime=start_time, endTime=end_time, options=PLATE_EXPECTED_YES)
    await lift_mover.Move(LIFT_POS['home'])

    return (time.perf_counter() - tickstart)


async def stk_put_plate_all():
    results = await asyncio.gather(
        stk_put_plate('left'),
        stk_put_plate('right'),
    )
    return results


async def stk_put_plate(side):
    duration = 0.0
    duration += await stk_put_p1(side)
    duration += await stk_put_p2(side)
    return f"stk_put_plate() done after {duration:.3f}s"
    
    
async def stk_put_p1(side):
    lift_mover = get_lift_mover(side)

    tickstart = time.perf_counter()

    await send_msg(json.dumps({'result': f"Put {side} plate phase 1..."}))

    # Check that the lift mover is at the home position.    #TODO A more sophisticated method to ensure the correct stacker state is adviced.
    (position,) = await lift_mover.GetPosition()
    if position != LIFT_POS['home']:
        raise Exception(f"Lift mover not at home position")
    
    # Lift the plate to waiting phase.
    await lift_mover.UseProfile(LIFT_PROFILE)
    start_time = 0
    end_time   = calculate_lift_move_time(LIFT_POS['phase'] - LIFT_POS['home'])
    await lift_mover.SetupPlateScan(startTime=start_time, endTime=end_time, options=PLATE_EXPECTED_YES)
    await lift_mover.Move(LIFT_POS['phase'])

    return (time.perf_counter() - tickstart)


async def stk_put_p2(side):
    mc6_stk = get_node_endpoint('mc6_stk')
    lift_mover = get_lift_mover(side)
    release_mover = get_release_mover(side)
    gripper_pwm = get_gripper_pwm(side)

    tickstart = time.perf_counter()

    await send_msg(json.dumps({'result': f"Put {side} plate phase 2..."}))
    
    # Check that the lift mover is at the phase position.   #TODO A more sophisticated method to ensure the correct stacker state is adviced.
    (position,) = await lift_mover.GetPosition()
    if position != LIFT_POS['phase']:
        raise Exception(f"Lift mover not at phase position.")

    # Put the hooks to semiopen state above the plate's edge to ensure that the plate can be put to stack and other don't come tumbling down.
    if not stk_get_stack_sensor_status(side):
        raise Exception("No stack present.")

    await release_mover.Move(RELEASE_POS['half_open'])
    
    # Get a grip from the plate to stabilize it -> Gripper Close.
    await mc6_stk.EnablePWMOutput(gripper_pwm, 1)

    # Lift the plate below the stack.
    await lift_mover.UseProfile(LIFT_PROFILE)
    await lift_mover.Move(LIFT_POS['grip'] - PLATE_HEIGHT - GRIP_SAFETY_MARGIN)
    
    # Lift the plate halfway into the stack.
    await lift_mover.UseProfile(LIFT_SLOW_PROFILE)
    await lift_mover.Move(LIFT_POS['grip'] - PLATE_EDGE_HEIGHT)
    
    # Gripper Open.
    await mc6_stk.EnablePWMOutput(gripper_pwm, 0)
    # Open the hooks in the stack.
    await release_mover.Move(RELEASE_POS['open'])
    # Lift the plate fully into the stack.
    await lift_mover.Move(LIFT_POS['lift_stack'])
    # Close the hooks in the stack.
    await release_mover.Move(RELEASE_POS['lock'])

    # Place the stack onto the hooks.
    await lift_mover.Move(LIFT_POS['grip'] - GRIP_SAFETY_MARGIN)
    
    # Check that the plate stayed in the stack while going home.
    await lift_mover.UseProfile(LIFT_PROFILE)
    start_time = calculate_lift_move_time(LIFT_POS['lift_stack'] - LIFT_POS['phase'])
    end_time   = calculate_lift_move_time(LIFT_POS['lift_stack'] - LIFT_POS['home'])
    await lift_mover.SetupPlateScan(startTime=start_time, endTime=end_time, options=PLATE_EXPECTED_NO)
    await lift_mover.Move(LIFT_POS['home'])

    return (time.perf_counter() - tickstart)

